<?php

namespace App\Http\Controllers;

use App\Models\Doctor;
use App\Models\Branch;
use App\Models\User;
use App\Models\Prescription;
use App\Models\Appointment;
use App\Models\Patient;
use App\Models\DoctorAvailability;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{
    public function dashboard()
    {
        $user = Auth::user();
        $stats = [
            'doctors' => Doctor::count(),
            'verified_doctors' => Doctor::where('is_verified', true)->count(),
            'pending_doctors' => Doctor::where('is_verified', false)->count(),
            'branches' => Branch::count(),
            'prescriptions' => Prescription::count(),
            'appointments' => Appointment::count(),
            'pending_appointments' => Appointment::where('status', 'pending')->count(),
            'patients' => Patient::count(),
        ];

        $notifications = $user->unreadNotifications()->latest()->take(10)->get();

        return view('admin.dashboard', compact('stats', 'notifications'));
    }

    public function doctors(Request $request)
    {
        $query = Doctor::with('user');

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('practice_number', 'like', "%{$search}%")
                  ->orWhere('specialization', 'like', "%{$search}%")
                  ->orWhere('qualification', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'pending') {
                $query->where('is_verified', false);
            } elseif ($status === 'verified') {
                $query->where('is_verified', true);
            } elseif ($status === 'active') {
                $query->where('is_verified', true)->where('is_active', true);
            } elseif ($status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        // Sort functionality
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        if ($sortBy === 'name') {
            $query->orderBy('first_name', $sortOrder)
                  ->orderBy('last_name', $sortOrder);
        } elseif ($sortBy === 'email') {
            $query->orderBy('email', $sortOrder);
        } elseif ($sortBy === 'practice_number') {
            $query->orderBy('practice_number', $sortOrder);
        } elseif ($sortBy === 'status') {
            $query->orderBy('is_verified', $sortOrder === 'asc' ? 'desc' : 'asc')
                  ->orderBy('is_active', $sortOrder === 'asc' ? 'desc' : 'asc');
        } else {
            $query->orderBy($sortBy, $sortOrder);
        }

        $doctors = $query->get();
        
        // Get filter counts for display
        $totalCount = Doctor::count();
        $pendingCount = Doctor::where('is_verified', false)->count();
        $verifiedCount = Doctor::where('is_verified', true)->count();
        $activeCount = Doctor::where('is_verified', true)->where('is_active', true)->count();
        $inactiveCount = Doctor::where('is_active', false)->count();

        return view('admin.doctors', compact('doctors', 'totalCount', 'pendingCount', 'verifiedCount', 'activeCount', 'inactiveCount'));
    }

    public function createDoctor()
    {
        return view('admin.create-doctor');
    }

    public function storeDoctor(Request $request)
    {
        $validated = $request->validate([
            'title' => 'nullable|string',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'specialization' => 'nullable|string',
            'qualification' => 'nullable|string',
            'practice_number' => 'required|string|unique:doctors,practice_number',
            'country' => 'nullable|string',
            'educational_details' => 'nullable|string',
            'phone' => 'nullable|string',
            'email' => 'required|email|unique:users,email|unique:doctors,email',
            'password' => 'required|string|min:8|confirmed',
            'profile_picture' => 'nullable|image|max:2048',
            'letterhead' => 'nullable|image|max:2048',
            'stamp' => 'nullable|image|max:2048',
            'signature' => 'nullable|image|max:2048',
            'is_verified' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
        ]);

        // Create user
        $user = User::create([
            'name' => "{$validated['first_name']} {$validated['last_name']}",
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => 'doctor',
            'is_active' => $validated['is_active'] ?? true,
        ]);

        // Handle file uploads
        $profilePicturePath = null;
        $letterheadPath = null;
        $stampPath = null;
        $signaturePath = null;

        if ($request->hasFile('profile_picture')) {
            $profilePicturePath = $request->file('profile_picture')->store('doctors/profile-pictures', 'public');
        }

        if ($request->hasFile('letterhead')) {
            $letterheadPath = $request->file('letterhead')->store('doctors/letterheads', 'public');
        }

        if ($request->hasFile('stamp')) {
            $stampPath = $request->file('stamp')->store('doctors/stamps', 'public');
        }

        if ($request->hasFile('signature')) {
            $signaturePath = $request->file('signature')->store('doctors/signatures', 'public');
        }

        // Create doctor profile
        $doctor = Doctor::create([
            'user_id' => $user->id,
            'title' => $validated['title'],
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'specialization' => $validated['specialization'],
            'qualification' => $validated['qualification'],
            'practice_number' => $validated['practice_number'],
            'country' => $validated['country'] ?? 'Lesotho',
            'educational_details' => $validated['educational_details'],
            'phone' => $validated['phone'],
            'email' => $validated['email'],
            'profile_picture_path' => $profilePicturePath,
            'letterhead_path' => $letterheadPath,
            'stamp_path' => $stampPath,
            'signature_path' => $signaturePath,
            'is_verified' => $validated['is_verified'] ?? true,
            'is_active' => $validated['is_active'] ?? true,
            'verified_at' => $validated['is_verified'] ?? true ? now() : null,
        ]);

        return redirect()->route('admin.doctors')
            ->with('success', 'Doctor registered successfully.');
    }

    public function showDoctor($id)
    {
        $doctor = Doctor::with('user')->findOrFail($id);
        return view('admin.doctor-details', compact('doctor'));
    }

    public function editDoctor($id)
    {
        $doctor = Doctor::with('user')->findOrFail($id);
        return view('admin.edit-doctor', compact('doctor'));
    }

    public function updateDoctor(Request $request, $id)
    {
        $doctor = Doctor::with('user')->findOrFail($id);
        
        $validated = $request->validate([
            'title' => 'nullable|string',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'specialization' => 'nullable|string',
            'qualification' => 'nullable|string',
            'practice_number' => 'required|string|unique:doctors,practice_number,' . $doctor->id,
            'country' => 'nullable|string',
            'educational_details' => 'nullable|string',
            'phone' => 'nullable|string',
            'email' => 'required|email|unique:users,email,' . $doctor->user_id . '|unique:doctors,email,' . $doctor->id,
            'password' => 'nullable|string|min:8|confirmed',
            'profile_picture' => 'nullable|image|max:2048',
            'letterhead' => 'nullable|image|max:2048',
            'stamp' => 'nullable|image|max:2048',
            'signature' => 'nullable|image|max:2048',
            'is_verified' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
        ]);

        // Update user
        $userData = [
            'name' => "{$validated['first_name']} {$validated['last_name']}",
            'email' => $validated['email'],
            'is_active' => $validated['is_active'] ?? $doctor->user->is_active,
        ];

        if (!empty($validated['password'])) {
            $userData['password'] = Hash::make($validated['password']);
        }

        $doctor->user->update($userData);

        // Handle file uploads
        if ($request->hasFile('profile_picture')) {
            if ($doctor->profile_picture_path) {
                Storage::disk('public')->delete($doctor->profile_picture_path);
            }
            $validated['profile_picture_path'] = $request->file('profile_picture')->store('doctors/profile-pictures', 'public');
        }

        if ($request->hasFile('letterhead')) {
            if ($doctor->letterhead_path) {
                Storage::disk('public')->delete($doctor->letterhead_path);
            }
            $validated['letterhead_path'] = $request->file('letterhead')->store('doctors/letterheads', 'public');
        }

        if ($request->hasFile('stamp')) {
            if ($doctor->stamp_path) {
                Storage::disk('public')->delete($doctor->stamp_path);
            }
            $validated['stamp_path'] = $request->file('stamp')->store('doctors/stamps', 'public');
        }

        if ($request->hasFile('signature')) {
            if ($doctor->signature_path) {
                Storage::disk('public')->delete($doctor->signature_path);
            }
            $validated['signature_path'] = $request->file('signature')->store('doctors/signatures', 'public');
        }

        // Update doctor profile
        $doctorData = [
            'title' => $validated['title'],
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'specialization' => $validated['specialization'],
            'qualification' => $validated['qualification'],
            'practice_number' => $validated['practice_number'],
            'country' => $validated['country'] ?? 'Lesotho',
            'educational_details' => $validated['educational_details'],
            'phone' => $validated['phone'],
            'email' => $validated['email'],
            'is_verified' => $validated['is_verified'] ?? $doctor->is_verified,
            'is_active' => $validated['is_active'] ?? $doctor->is_active,
        ];

        if (isset($validated['profile_picture_path'])) {
            $doctorData['profile_picture_path'] = $validated['profile_picture_path'];
        }
        if (isset($validated['letterhead_path'])) {
            $doctorData['letterhead_path'] = $validated['letterhead_path'];
        }
        if (isset($validated['stamp_path'])) {
            $doctorData['stamp_path'] = $validated['stamp_path'];
        }
        if (isset($validated['signature_path'])) {
            $doctorData['signature_path'] = $validated['signature_path'];
        }

        if (($validated['is_verified'] ?? $doctor->is_verified) && !$doctor->verified_at) {
            $doctorData['verified_at'] = now();
        }

        $doctor->update($doctorData);

        return redirect()->route('admin.doctors.show', $doctor->id)
            ->with('success', 'Doctor updated successfully.');
    }

    public function verifyDoctor($id)
    {
        $doctor = Doctor::findOrFail($id);
        $doctor->update([
            'is_verified' => true,
            'is_active' => true,
            'verified_at' => now(),
        ]);

        return back()->with('success', 'Doctor verified successfully.');
    }

    public function toggleDoctorStatus($id)
    {
        $doctor = Doctor::findOrFail($id);
        $doctor->update(['is_active' => !$doctor->is_active]);

        return back()->with('success', 'Doctor status updated.');
    }

    public function deleteDoctor($id)
    {
        $doctor = Doctor::findOrFail($id);
        $doctor->user->delete();

        return back()->with('success', 'Doctor deleted successfully.');
    }

    public function branches()
    {
        $branches = Branch::latest()->get();
        return view('admin.branches', compact('branches'));
    }

    public function storeBranch(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string',
            'phone' => 'nullable|string',
            'email' => 'nullable|email',
        ]);

        Branch::create($validated);

        return back()->with('success', 'Branch created successfully.');
    }

    public function updateBranch(Request $request, $id)
    {
        $branch = Branch::findOrFail($id);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string',
            'phone' => 'nullable|string',
            'email' => 'nullable|email',
        ]);

        $branch->update($validated);

        return back()->with('success', 'Branch updated successfully.');
    }

    public function deleteBranch($id)
    {
        Branch::findOrFail($id)->delete();
        return back()->with('success', 'Branch deleted successfully.');
    }

    public function prescriptions()
    {
        $prescriptions = Prescription::with(['patient', 'doctor', 'appointment'])
            ->latest()
            ->get();

        return view('admin.prescriptions', compact('prescriptions'));
    }

    public function deletePrescription($id)
    {
        $prescription = Prescription::findOrFail($id);
        
        // Delete PDF if exists
        if ($prescription->pdf_path && Storage::disk('public')->exists($prescription->pdf_path)) {
            Storage::disk('public')->delete($prescription->pdf_path);
        }
        
        $prescription->delete();

        return redirect()->route('admin.prescriptions')
            ->with('success', 'Prescription deleted successfully.');
    }

    public function appointments()
    {
        $appointments = Appointment::with(['patient', 'doctor', 'branch', 'pharmacist'])
            ->latest()
            ->get();

        return view('admin.appointments', compact('appointments'));
    }

    public function patients()
    {
        $patients = Patient::withCount(['appointments', 'prescriptions'])
            ->latest()
            ->get();

        return view('admin.patients', compact('patients'));
    }

    public function notifications()
    {
        $user = Auth::user();
        $notifications = $user->notifications()->latest()->paginate(20);
        return view('admin.notifications', compact('notifications'));
    }

    public function markNotificationRead($id)
    {
        $notification = Auth::user()->notifications()->findOrFail($id);
        $notification->markAsRead();
        return back()->with('success', 'Notification marked as read.');
    }

    public function markAllNotificationsRead()
    {
        Auth::user()->unreadNotifications->markAsRead();
        return back()->with('success', 'All notifications marked as read.');
    }

    public function viewDoctorAvailability($id)
    {
        $doctor = Doctor::findOrFail($id);
        $availabilities = DoctorAvailability::where('doctor_id', $doctor->id)
            ->orderByRaw("FIELD(day_of_week, 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday')")
            ->orderBy('start_time')
            ->get();
        
        return view('admin.doctor-availability', compact('doctor', 'availabilities'));
    }
}

